import { useQuery } from "@tanstack/react-query";
import { InfoIcon, MapPinIcon, TrendingUpIcon, PlaneIcon } from "lucide-react";
import StatCard from "@/components/StatCard";
import ChartCard from "@/components/ChartCard";
import { extractChartData } from "@/lib/utils";
import { Skeleton } from "@/components/ui/skeleton";
import type { Jump } from "@shared/schema";

export default function Dashboard() {
  // Fetch statistics
  const { data: stats, isLoading: statsLoading } = useQuery<{
    totalJumps: number;
    dropzones: number;
    maxAltitude: string;
    avgAltitude: string;
    aircraftTypes: number;
    topDropzone: string;
    topAircraft: string;
    monthlyIncrease: number;
  }>({
    queryKey: ['/api/stats'],
  });

  // Fetch all jumps for charts
  const { data: jumpsData, isLoading: jumpsLoading } = useQuery<{ jumps: Jump[], total: number }>({
    queryKey: ['/api/jumps'],
  });

  // Extract jumps array from the response
  const jumps = jumpsData?.jumps || [];

  // Extract chart data from jumps
  const dropzoneChartData = jumps.length > 0 ? extractChartData(jumps, 'dropzone') : { labels: [], data: [] };
  const aircraftChartData = jumps.length > 0 ? extractChartData(jumps, 'aircraft') : { labels: [], data: [] };
  const equipmentChartData = jumps.length > 0 ? extractChartData(jumps, 'equipment') : { labels: [], data: [] };
  const jumpTypeChartData = jumps.length > 0 ? extractChartData(jumps, 'jumpType') : { labels: [], data: [] };

  return (
    <section className="mb-12">
      <h2 className="text-2xl font-bold text-gray-800 mb-6" style={{ fontFamily: "Montserrat, sans-serif" }}>Jump Statistics</h2>
      
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
        {statsLoading ? (
          <>
            <Skeleton className="h-36 w-full" />
            <Skeleton className="h-36 w-full" />
            <Skeleton className="h-36 w-full" />
            <Skeleton className="h-36 w-full" />
          </>
        ) : (
          <>
            <StatCard
              title="Total Jumps"
              value={stats?.totalJumps || 0}
              icon={InfoIcon}
              iconColor="text-[#3B82F6]"
              iconBgColor="bg-[#3B82F6]"
              additionalInfo={
                <span className="text-sm text-green-600 font-medium flex items-center">
                  <TrendingUpIcon className="h-4 w-4 mr-1" />
                  <span>{stats?.monthlyIncrease || 0}%</span> from last month
                </span>
              }
            />
            
            <StatCard
              title="Different Dropzones"
              value={stats?.dropzones || 0}
              icon={MapPinIcon}
              iconColor="text-[#F97316]"
              iconBgColor="bg-[#F97316]"
              additionalInfo={
                <span className="text-sm text-gray-600 font-medium">
                  Most jumps at: <span className="font-semibold text-[#F97316]">{stats?.topDropzone || "None"}</span>
                </span>
              }
            />
            
            <StatCard
              title="Max Altitude"
              value={stats?.maxAltitude || "0 ft"}
              icon={TrendingUpIcon}
              iconColor="text-[#10B981]"
              iconBgColor="bg-[#10B981]"
              additionalInfo={
                <span className="text-sm text-gray-600 font-medium">
                  Average altitude: <span className="font-semibold text-[#10B981]">{stats?.avgAltitude || "0 ft"}</span>
                </span>
              }
            />
            
            <StatCard
              title="Aircraft Types"
              value={stats?.aircraftTypes || 0}
              icon={PlaneIcon}
              iconColor="text-[#1E40AF]"
              iconBgColor="bg-[#1E40AF]"
              additionalInfo={
                <span className="text-sm text-gray-600 font-medium">
                  Most used: <span className="font-semibold text-[#1E40AF]">{stats?.topAircraft || "None"}</span>
                </span>
              }
            />
          </>
        )}
      </div>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {jumpsLoading ? (
          <>
            <Skeleton className="h-96 w-full" />
            <Skeleton className="h-96 w-full" />
            <Skeleton className="h-96 w-full" />
            <Skeleton className="h-96 w-full" />
          </>
        ) : (
          <>
            <ChartCard
              title="Jumps by Dropzone"
              type="bar"
              labels={dropzoneChartData.labels}
              data={dropzoneChartData.data}
              backgroundColor="#3B82F6"
              borderColor="#1E40AF"
            />
            
            <ChartCard
              title="Jumps by Aircraft"
              type="pie"
              labels={aircraftChartData.labels}
              data={aircraftChartData.data}
              backgroundColor={[
                '#3B82F6',
                '#1E40AF',
                '#60A5FA',
                '#93C5FD',
                '#BFDBFE'
              ]}
            />
            
            <ChartCard
              title="Jumps by Equipment"
              type="doughnut"
              labels={equipmentChartData.labels}
              data={equipmentChartData.data}
              backgroundColor={[
                '#F97316',
                '#FB923C',
                '#FDBA74',
                '#FED7AA',
                '#FFEDD5'
              ]}
            />
            
            <ChartCard
              title="Jumps by Type"
              type="bar"
              labels={jumpTypeChartData.labels}
              data={jumpTypeChartData.data}
              backgroundColor="#10B981"
              borderColor="#059669"
            />
          </>
        )}
      </div>
    </section>
  );
}
